##################################################################
# Description:   Talking like a normal user.                     #
# Author:        Norbert Kuemin <norbert.kuemin@gmx.net>         #
# Copyright:     Copyright 2001 by Norbert Kuemin                #
##################################################################
# Version   Date         Description                             #
#    1.00   02.06.2001   Initial coding                          #
##################################################################

package modules::talk;

$VERSION = "1.00";
$NAME = "Talk";
$MODULE = "modules::talk";

sub new {
   my $proto = shift;

   my $conn = undef;
   if (scalar(@_) > 0) {
      $conn = shift;
   }

   my $config = undef;
   if (scalar(@_) > 0) {
      $config = shift;
   }

   my $eventhandler = undef;
   if (scalar(@_) > 0) {
      $eventhandler = shift;
   }

   my $modulehandler = undef;
   if (scalar(@_) > 0) {
      $modulehandler = shift;
   }

   my $self = {
      '_config'        => $config,
      '_connection'    => $conn,
      '_eventhandler'  => $eventhandler,
      '_modulehandler' => $modulehandler,
      '_events'        => 'msg public join part action mail away_on away_off',
      '_delimiter'     => "\t",
      '_commonfile'    => undef,
      '_questionfile'  => undef,
      '_answerfile'    => undef,
      '_commonlist'    => undef,
      '_questionlist'  => undef,
      '_answerlist'    => undef,
      '_userfile'      => undef,
      '_uservar'       => undef,
      '_feelfile'      => undef,
      '_feellist'      => undef,
      '_actionfile'    => undef,
      '_actionlist'    => undef,
      '_mailchannel'   => undef,
      '_away'          => 0
   };

   bless $self, $proto;

   if (defined($config)) {
      if (defined($config->get('TALKLIB'))) {
         $self->{'_commonfile'} = $config->get('TALKLIB');
      }
      if (defined($config->get('TALKLIBQ'))) {
         $self->{'_questionfile'} = $config->get('TALKLIBQ');
      }
      if (defined($config->get('TALKLIBA'))) {
         $self->{'_answerfile'} = $config->get('TALKLIBA');
      }
      if (defined($config->get('TALKUSER'))) {
         $self->{'_userfile'} = $config->get('TALKUSER');
      }
      if (defined($config->get('TALKFEEL'))) {
         $self->{'_feelfile'} = $config->get('TALKFEEL');
      }
      if (defined($config->get('TALKACTION'))) {
         $self->{'_actionfile'} = $config->get('TALKACTION');
      }
      if (defined($config->get('TALKMAIL'))) {
         $self->{'_mailchannel'} = $config->get('TALKMAIL');
      }
   }

   $self->load();
   $self->user_load();
   $self->feel_load();
   $self->action_load();

   return $self;
}

sub on_away_on {
   $self = shift;
   $self->{'_away'} = 1;
}

sub on_away_off {
   $self = shift;
   $self->{'_away'} = 0;
}

sub on_join {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   $self->user_set($nick,"hello","0");
   $self->user_set($nick,"bye","0");
   $self->user_save();
}

sub on_part {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   $self->user_set($nick,"hello","0");
   $self->user_set($nick,"bye","0");
   $self->user_save();
}

sub on_msg {
   $self = shift;
   my ($nick, $userhost, $text) = @_;
   my $cmd = uc($self->{'_config'}->get('COMMANDPREFIX')."TALK");
   my $search = substr(uc($text),0,length($cmd));
   if ($search eq $cmd) {
      $self->command($nick,split(/ /,$text));
   }
}


sub on_public {
   $self = shift;
   my ($channel, $nick, $userhost, $text) = @_;

   # No talk to myself
   if ($nick eq $self->{'_connection'}->nick()) { return; }

   # No talk if i'm away
   if ($self->{'_away'} > 0) { return; }

   # Init
   my $tome = 0;

   # Analyse incoming messages
   my $type = $self->analyze_text($channel, $nick, $userhost, $text);

   # Do we ignore the user??
   if ($self->user_get($nick,"ignore") > time()) { return; }

   if ($type eq "") { return; }

   # say hello only once per session
   if (($type eq "hello") || ($type eq "p_hello")) {
      if ($self->user_get($nick,"hello") eq "1") {
         return;
      } else {
         $self->user_set($nick,"hello","1");
         $self->user_save();
      }
   }
   # say bye only once per session
   if (($type eq "bye") || ($type eq "p_bye")) {
      if ($self->user_get($nick,"bye") eq "1") {
         return;
      } else {
         $self->user_set($nick,"bye","1");
         $self->user_save();
      }
   }

   # Answer if something found
   my $answer = $self->get_text($type);
   if (length($answer) > 0) {
      my %fieldlist = (
         'NICK' => $nick,
         'CHANNEL' => $channel,
         'USERHOST' => $userhost,
         'NAME' => $self->{'_connection'}->nick()
      );
      $answer = $self->replace_field($answer,%fieldlist);
      if ($tome > 0) { $answer = $nick.": ".$answer; }
      $self->say($channel,$answer);
   }
   $self->feel_do($type,$nick);
}

sub on_action {
   $self = shift;
   ($channel, $nick, $userhost, $action) = @_;
   if (index(lc($action),lc($self->{'_connection'}->nick())) != -1) {
      # Log unknown actions about me
      $self->write_log($channel,$nick,$action,2);
   }
}

sub on_mail {
   $self = shift;
   ($email, $name, $subject, $text) = @_;
   if (defined($self->{'_mailchannel'})) {
      my $reptext = $self->get_text("mail");
      $self->say($self->{'_mailchannel'},$reptext);
   }
}

sub get_version {
   return $VERSION;
}

sub get_name {
   return $NAME;
}

sub get_module {
   return $MODULE;
}

sub get_events {
   $self = shift;
   return $self->{'_events'};
}

sub analyze_text {
   $self = shift;
   my ($channel, $nick, $userhost, $text) = @_;
   my $type = "";
   my $botname = lc($self->{'_connection'}->nick());
   if ($botname eq lc(substr($text,0,length($botname)))) {
      $tome = 1;
      my $ntext = substr($text,0-length($text)+length($self->{'_connection'}->nick()));
      if ($self->clean_text($ntext) eq "") {
         $type = "general";
      } else {
         $type = $self->analyze_common_text($ntext);
         if ($type eq "") {
            # Question text ?
            $type = $self->analyze_question_text($ntext);
         } else {
            if (lc(substr($type,0,2)) ne "p_") {
               if ($self->get_text("p_".$type) ne "") { $type = "p_".$type; }
            }
         }
         if ($type eq "") {
            # Log unknown messages
            $self->write_log($channel,$nick,$text,1);
         }
      }
   } else {
      $type = $self->analyze_common_text($text);
      if ($type ne "") {
         if (lc(substr($type,0,2)) ne "p_") {
            if ($self->get_text("p_".$type) ne "") { $type = "p_".$type; }
         }
      } else {
         if (index(lc($text),lc($self->{'_connection'}->nick())) != -1) {
            # Log unknown messages about me
            $self->write_log($channel,$nick,$text,1);
         }
      }
   }
   return $type;
}

sub analyze_common_text {
   $self = shift;
   my ($text) = @_;
   my %fieldlist = (
      'NICK' => $self->{'_connection'}->nick(),
   );
   $text = $self->clean_text($text);
   *list = $self->{'_commonlist'};
   foreach $type (keys %list) {
      @textlist = split($self->{'_delimiter'},$self->{'_commonlist'}->{$type});
      foreach (@textlist) {
         $_ = $self->replace_field($_,%fieldlist);
         if (lc($text) eq lc($_)) { return $type; }
      }
   }
   return "";
}   

sub analyze_question_text {
   $self = shift;
   my ($text) = @_;
   my %fieldlist = (
      'NICK' => $self->{'_connection'}->nick(),
   );
   $text = $self->clean_text($text);
   *list = $self->{'_questionlist'};
   foreach $type (keys %list) {
      @textlist = split($self->{'_delimiter'},$self->{'_questionlist'}->{$type});
      foreach (@textlist) {
         $_ = $self->replace_field($_,%fieldlist);
         if (lc($text) eq lc($_)) { return $type; }
      }
   }
   return "";
}   

sub clean_text {
   $self = shift;
   my ($text) = @_;
   $_ = $text;
   foreach $ch ("\\.",",","!","\\?",":",";","\\/","\\(","\\)") {
      s/$ch//g;
   }
   s/^\s*//;
   s/\s*$//;
   return $_;
}

sub get_text {
   $self = shift;
   my $type = shift;
   if (defined($self->{'_commonlist'}->{$type})) {
      my @list = split($self->{'_delimiter'},$self->{'_commonlist'}->{$type});
      my $entry = rand scalar(@list);
      $entry = int $entry;
      return $list[$entry];
   }
   if (defined($self->{'_answerlist'}->{$type})) {
      my @list = split($self->{'_delimiter'},$self->{'_answerlist'}->{$type});
      my $entry = rand scalar(@list);
      $entry = int $entry;
      return $list[$entry];
   }
   print "Talk: No entry found for '$type'.\n";
   return "";
}

sub replace_field {
   $self = shift;
   my ($text,%fieldlist) = @_;
   $_ = $text;
   foreach $name (keys %fieldlist) {
      my $fname = "<".$name.">";
      s/$fname/$fieldlist{$name}/g;
   }
   return $_;
}

sub load {
   $self = shift; 
   # Loading data
   if (-r $self->{'_commonfile'}) {
      open FD, $self->{'_commonfile'} or die "Seen: Error reading '".$self->{'_commonfile'}."'.\n";
      while (<FD>) {
         chomp;
         my ($key,$data) = split($self->{'_delimiter'},$_,2);
         $self->{'_commonlist'}->{$key} = $data;
      }
      close FD;   
   }
   if (-r $self->{'_questionfile'}) {
      open FD, $self->{'_questionfile'} or die "Seen: Error reading '".$self->{'_questionfile'}."'.\n";
      while (<FD>) {
         chomp;
         my ($key,$data) = split($self->{'_delimiter'},$_,2);
         $self->{'_questionlist'}->{$key} = $data;                                               
      }
      close FD;   
   }
   if (-r $self->{'_answerfile'}) {
      open FD, $self->{'_answerfile'} or die "Seen: Error reading '".$self->{'_answerfile'}."'.\n";
      while (<FD>) {
         chomp;
         my ($key,$data) = split($self->{'_delimiter'},$_,2);
         $self->{'_answerlist'}->{$key} = $data;                                               
      }
      close FD;   
   }
}

sub unload {
   $self = shift;
   $self->{'_commonlist'} = undef;
   $self->{'_questionlist'} = undef;
   $self->{'_answerlist'} = undef;
   $self->{'_uservar'} = undef;
   $self->{'_feellist'} = undef;
   $self->{'_actionlist'} = undef;
}

sub say {
   $self = shift;
   my ($to,$text) = @_;
   $self->{'_connection'}->privmsg($to,$text);
}

sub write_log {
   $self = shift;
   my ($channel,$nick,$text,$type) = @_;
   if ($self->{'_config'}->get('TALKLOG') ne "") {
      open FD, ">>".$self->{'_config'}->get('TALKLOG');
      if ($type == 1) {
         print FD localtime()."\t".$channel."\t".$nick."\t".$text."\n";
      } else {
         print FD localtime()."\t".$channel."\t*".$nick." ".$text."\n";
      }
      close FD;
   }
}

sub command {
   $self = shift;
   my ($nick, @cmd) = @_;
   my $userhandler = $self->{'_modulehandler'}->get("modules::user");
   if (defined($userhandler)) {
      if ($userhandler->checkmode(lc($nick),"m") == 0) { return; }
   }
   if (uc($cmd[1]) eq "RELOAD") {
      $self->unload();
      $self->say($nick, "All tables unloaded.");
      $self->load();
      $self->say($nick, "New talk tables loaded.");
      $self->user_load();
      $self->say($nick, "New user table loaded.");
      $self->feel_load();
      $self->say($nick, "New feeling table loaded."); 
      $self->action_load();
      $self->say($nick, "New action table loaded."); 
   }
   if (uc($cmd[1]) eq "IGNORE") {
      my $itime = time()+$cmd[3];
      $self->user_set($cmd[2],"ignore",$itime);
      $self->user_save();
      if (scalar(@cmd) > 4) {
         $action = $self->action_get("p_ignore");
         if (length($action) > 0) {
            my %fieldlist = (
               'NICK' => $cmd[2],
               'CHANNEL' => $cmd[4],
            );
            $action = $self->replace_field($action,%fieldlist);
            $self->{'_connection'}->me($cmd[4], $action);
         }
      }
   }
}

sub user_load {
   $self = shift;
   # Loading user data
   if (-r $self->{'_userfile'}) {
      open FD, $self->{'_userfile'} or die "Talk: Error reading '".$self->{'_userfile'}."'.\n";
      while (<FD>) {
         chomp;
         my ($key,$data) = split($self->{'_delimiter'},$_,2);
         $self->{'_uservar'}->{$key} = $data;
      }
      close FD;
   }
}

sub user_save {
   $self = shift;
   *list = $self->{'_uservar'};
   open FD, ">".$self->{'_userfile'} or die "Talk: Error writing '".$self->{'_questionfile'}."'.\n";
   foreach $user (keys %list) {
      print FD $user.$self->{'_delimiter'}.$self->{'_uservar'}->{$user}."\n";
   }
   close FD;
}

sub user_get {
   $self = shift;
   my ($user, $what) = @_;
   $user = lc($user);
   $what = lc($what);
   if (defined($self->{'_uservar'}->{$user})) {
      my @value = split($self->{'_delimiter'},$self->{'_uservar'}->{$user});
      if ($what eq "friendlevel") { return $value[0]; }
      if ($what eq "hello") { return $value[1]; }
      if ($what eq "bye") { return $value[2]; }
      if ($what eq "ignore") { return $value[3]; }
   }
   return "";
}

sub user_set {
   $self = shift;
   my ($user, $what, $newval) = @_;
   $user = lc($user);
   $what = lc($what);
   my @value = ("0","0","0","0");
   if (defined($self->{'_uservar'}->{$user})) {
      @value = split($self->{'_delimiter'},$self->{'_uservar'}->{$user});
   }
   if ($what eq "friendlevel") { $value[0] = $newval; }
   if ($what eq "hello") { $value[1] = $newval; }
   if ($what eq "bye") { $value[2] = $newval; }
   if ($what eq "ignore") { $value[3] = $newval; }
   $self->{'_uservar'}->{$user} = join($self->{'_delimiter'},@value);
}

sub feel_load {
   $self = shift;
   # Loading user data
   if (-r $self->{'_feelfile'}) {
      open FD, $self->{'_feelfile'} or die "Talk: Error reading '".$self->{'_feelfile'}."'.\n";
      while (<FD>) {
         chomp;
         my ($key,$data) = split($self->{'_delimiter'},$_,2);
         $self->{'_feellist'}->{$key} = $data;
      }
      close FD;
   }
}


sub feel_do {
   $self = shift;
   my ($type, $nick) = @_;
   if (defined($self->{'_feellist'}->{$type})) {
      my ($addval, $maxval) = split($self->{'_delimiter'},$self->{'_feellist'}->{$type});
      my $userval = $self->user_get($nick,"friendlevel");
      if (($addval > 0) && (($userval+$addval) <= $maxval)) {
         $userval = $userval + $addval;
         $self->user_set($nick,"friendlevel",$userval);
         $self->user_save();
      }
      if (($addval < 0) && (($userval+$addval) >= $maxval)) {
         $userval = $userval + $addval;
         $self->user_set($nick,"friendlevel",$userval);
         $self->user_save();
      }
   }
}

sub action_load {
   $self = shift;
   # Loading user data
   if (-r $self->{'_actionfile'}) {
      open FD, $self->{'_actionfile'} or die "Talk: Error reading '".$self->{'_actionfile'}."'.\n";
      while (<FD>) {
         chomp;
         my ($key,$data) = split($self->{'_delimiter'},$_,2);
         $self->{'_actionlist'}->{$key} = $data;
      }
      close FD;
   }
}

sub action_get {
   $self = shift;
   my $type = shift;
   if (defined($self->{'_actionlist'}->{$type})) {
      my @list = split($self->{'_delimiter'},$self->{'_actionlist'}->{$type});
      my $entry = rand scalar(@list);
      $entry = int $entry;
      return $list[$entry];
   }
   return "";
}
