##################################################################
# Description:   Do some shell commands                          #
# Author:        Norbert Kuemin <norbert.kuemin@gmx.net>         #
# Copyright:     Copyright 2003 by Norbert Kuemin                #
##################################################################
# Version   Date         Description                             #
#    1.00   25.04.2003   Initial coding                          #
##################################################################

package modules::shellcmd;

use NoK::tools;

$VERSION = "1.00";
$NAME = "Shellcmd";
$MODULE = "modules::shellcmd";

sub new {
   my $proto = shift;

   my $conn = undef;
   if (scalar(@_) > 0) {
      $conn = shift;
   }

   my $config = undef;
   if (scalar(@_) > 0) {
      $config = shift;
   }

   my $eventhandler = undef;
   if (scalar(@_) > 0) {
      $eventhandler = shift;
   }

   my $modulehandler = undef;
   if (scalar(@_) > 0) {
      $modulehandler = shift;
   }

   my $self = {
         '_config'        => $config,
         '_connection'    => $conn,
         '_eventhandler'  => $eventhandler,
         '_modulehandler' => $modulehandler,
         '_events'        => 'public msg',
         '_filename'      => 'shellcmd.config',
         '_data'          => {},
         '_help'          => {},
         '_delimiter'     => "\t",
         '_tools'         => new NoK::tools,
         '_debug'         => 0
   };

   bless $self, $proto;

   if (defined($config)) {
      if (defined($config->get('SHELLCMDFILE'))) {
         $self->{'_filename'} = $config->get('SHELLCMDFILE');
      }
   }
   $self->load();

   return $self;
}

sub on_public {
   $self = shift;
   my ($channel, $nick, $userhost, $text) = @_;
   $self->shellcmd($channel, $text);
}

sub on_msg {
   $self = shift;
   my ($nick, $userhost, $text) = @_;
   $self->shellcmd($nick, $text);
   my $cmd = uc($self->{'_config'}->get('COMMANDPREFIX')."HELP");
   $search = substr(uc($text),0,length($cmd));
   if ($search eq $cmd) {
      $_ = substr($text,length($cmd));
      s/^\s+//;
      s/\s+$//;
      $self->help($nick,$_);
   }
}

sub get_version {
   return $VERSION;
}

sub get_name {
   return $NAME;
}

sub get_module {
   return $MODULE;
}

sub get_events {
   $self = shift;
   return $self->{'_events'};
}

sub load {
   $self = shift;
   # Loading data
   if (-r $self->{'_filename'}) {
      open FD, $self->{'_filename'} or die "Seen: Error reading '".$self->{'_filename'}."'.\n";
      if ($self->{'_debug'}) { print "Load: ".$self->{'_filename'}."\n"; }
      while (<FD>) {
         chomp;
         my ($cmd,$shellcmd,$help) = split( $self->{'_delimiter'});
         $self->{'_data'}->{$cmd} = $shellcmd;
         $self->{'_help'}->{$cmd} = $help;
      }
      close FD;
   }
}

sub shellcmd {
   $self = shift;
   my ($reply, $text) = @_;
   *DATA = $self->{'_data'};
   foreach my $key (keys %DATA) {
      my $cmd = uc($self->{'_config'}->get('COMMANDPREFIX').$key);
      my $search = substr(uc($text),0,length($cmd));
      if ($search eq $cmd) {
         $_ = substr($text,length($cmd));
         s/^\s+//;
         s/\s+$//;
         my $out = $self->do($DATA{$key},split(/ /));
         if (length($out) > $self->{'_config'}->get('SHELLMAX')) {
            $out = $self->{'_config'}->get('SHELLMAXANSWER');
         }
         my @lines = split(/\n/,$out);
         foreach my $line (@lines) {
            $self->{'_connection'}->privmsg($reply,$line);
         }
      }
   }
}

sub do {
   $self = shift;
   my $cmd = shift;
   my $allargs = join(" ",@_);
   if ($self->{'_debug'}) { print "Execute: ".$allargs."\n"; }
   $cmd =~ s/\$\*/$allargs/;
   if ($self->{'_debug'}) { print "Execute: ".$cmd."\n"; }
   if (scalar(@_) > 0) {
      for ($i = 1; $i <= scalar(@_); $i++) {
         $cmd =~ s/\$$i/$_[$i-1]/;
      }
   }
   if ($self->{'_debug'}) { print "Execute: ".$cmd."\n"; }
   open(FD,$cmd." |");
   $ret = "";
   while (<FD>) {
      $ret = $ret.$_;
   }
   close(FD);
   if ($self->{'_debug'}) { print "Execute output: ".$ret."\n"; }
   return $ret;
}

sub help {
   $self = shift;
   my $nick = shift;
   if (scalar(@_) > 0) {
      $cmd = lc(shift);
   }
   *DATA = $self->{'_data'};
   *HELPTEXT = $self->{'_help'};
   foreach my $key (keys %DATA) {
         my $text = substr($key."                ",0,15)."- ";
         if (defined($HELPTEXT{$key})) {
            $text = $text.$HELPTEXT{$key};
         } else {
            $text = $text." Shell command with no helptext.";
         }
         $self->{'_connection'}->privmsg($nick,$text);
   }
}
