##################################################################
# Description:   Logging nick data                               #
# Author:        Norbert Kuemin <norbert.kuemin@gmx.net>         #
# Copyright:     Copyright 2000 by Norbert Kuemin                #
##################################################################
# Saving ALL datas about a nick on a channel in a structured     #
# file. The delimiter is a tab and the structur is like this:    #
# nick   channel   userhost   start   end                        #
##################################################################
# Version   Date         Description                             #
#     1.0   23.07.2000   Initial coding                          #
##################################################################

package modules::seen;

use NoK::tools;

$VERSION = "1.0";
$NAME = "Seen";
$MODULE = "modules::seen";

sub new {
   my $proto = shift;

   my $conn = undef;
   if (scalar(@_) > 0) {
      $conn = shift;
   }

   my $config = undef;
   if (scalar(@_) > 0) {
      $config = shift;
   }

   my $eventhandler = undef;
   if (scalar(@_) > 0) {
      $eventhandler = shift;
   }

   my $modulehandler = undef;
   if (scalar(@_) > 0) {
      $modulehandler = shift;
   }

   my $self = {
         '_config'        => $config,
         '_connection'    => $conn,
         '_eventhandler'  => $eventhandler,
         '_modulehandler' => $modulehandler,
         '_events'        => 'join part quit nick names leaving public msg kick',
         '_filename'      => 'seen.list',
         '_myjoin'        => '',
         '_data'          => {},
         '_delimiter'     => "\t",
         '_tools'         => new NoK::tools,
         '_debug'         => 0
   };

   bless $self, $proto;

   if (defined($config)) {
      if (defined($config->get('SEENFILE'))) {
         $self->{'_filename'} = $config->get('SEENFILE');
      }
   }
   $self->load();

   return $self;
}

sub on_public {
   $self = shift;
   my ($channel, $nick, $userhost, $text) = @_;
   $self->seen($channel, $nick, $text);
}

sub on_msg {
   $self = shift;
   my ($nick, $userhost, $text) = @_;
   $self->seen("", $nick, $text);
}

sub on_join {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   my $mynick = "";
   if (defined($self->{'_config'})) {
      $mynick = $self->{'_connection'}->nick();
   }
   if ($nick eq $mynick) {
      $self->{'_myjoin'} = $channel;
      *DATA = $self->{'_data'};
      foreach my $key (keys %DATA) {
         my ($chan, $user) = split($self->{'_delimiter'},$key);
         if ($chan eq $channel) {
            my ($nuserhost, $x, $stop) = split($self->{'_delimiter'},$DATA{$key});
            if ($stop eq "") { 
               $self->stop($channel, $user, $nuserhost);
            }
         }
      }
      $self->save();
   } else {
      $self->start($channel, $nick, $userhost);
      $self->save();
   }
}

sub on_part {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   $self->stop($channel, $nick, $userhost);
   $self->save();
}

sub on_quit {
   $self = shift;
   my ($nick, $userhost) = @_;
   $self->stopall($nick, $userhost);
   $self->save();
}

sub on_nick {
   $self = shift;
   my ($nick, $userhost, $newnick) = @_;
   $self->stopstartall($nick, $userhost, $newnick);
   $self->save();
}

sub on_names {
   $self = shift;
   my ($channel, $list) = @_;
   if ($channel eq $self->{'_myjoin'}) {
      $self->{'_myjoin'} = '';
      foreach $nick (split(/ /,$list)) {
         if (length($nick)>0) { $self->start($channel, $nick, ""); }
      }
      $self->save();
   }
}

sub on_leaving {
   $self = shift;
   *DATA = $self->{'_data'};
   foreach my $key (keys %DATA) {
      my ($userhost, $x, $stop) = split( $self->{'_delimiter'},$DATA{$key});
      if (length($stop) == 0) {
         my ($channel,$nick)  = split( $self->{'_delimiter'},$key);
         $self->stop($channel,$nick,$userhost);
      }
   }
   $self->save();
}

sub on_kick {
   $self = shift;
   my ($channel, $nick, $userhost, $user, $reason) = @_;
   $self->stop($channel, $nick, $userhost);
}

sub get_version {
   return $VERSION;
}

sub get_name {
   return $NAME;
}

sub get_module {
   return $MODULE;
}

sub get_events {
   $self = shift;
   return $self->{'_events'};
}

sub load {
   $self = shift;
   # Loading data
   if (-r $self->{'_filename'}) {
      open FD, $self->{'_filename'} or die "Seen: Error reading '".$self->{'_filename'}."'.\n";
      if ($self->{'_debug'}) { print "Load: ".$self->{'_filename'}."\n"; }
      while (<FD>) {
         chomp;
         my ($channel,$nick,$userhost,$start,$stop) = split( $self->{'_delimiter'});
         my $key = $channel. $self->{'_delimiter'}.$nick;
         my $data = $userhost. $self->{'_delimiter'}.$start. $self->{'_delimiter'}.$stop;
         $self->{'_data'}->{$key} = $data;
      }
      close FD;
   }
}

sub save {
   $self = shift;
   open FD, ">".$self->{'_filename'};
   if ($self->{'_debug'}) { print "Save: ".$self->{'_filename'}."\n"; }
   *DATA = $self->{'_data'};
   foreach my $key (keys %DATA) {
      print FD $key. $self->{'_delimiter'}.$DATA{$key}."\n";
   }
   close FD;
}

sub start {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   $channel = lc($channel);
   my $nick = $self->{'_tools'}->realnick($nick);
   if ($nick ne $self->{'_connection'}->nick()) {
      my $start = time;
      my $key = $channel. $self->{'_delimiter'}.$nick;
      my $data = $userhost. $self->{'_delimiter'}.$start. $self->{'_delimiter'};
      $self->{'_data'}->{$key} = $data;
      if ($self->{'_debug'}) { print "Start: $key".$self->{'_delimiter'}.$data."\n"; }
   }
}

sub stop {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   $channel = lc($channel);
   my $stop = time;
   $nick = $self->{'_tools'}->realnick($nick);
   if ($nick ne $self->{'_connection'}->nick()) {
      my $key = $channel. $self->{'_delimiter'}.$nick;
      my ($x, $start) = split( $self->{'_delimiter'},$DATA{$key});
      my $data = $userhost. $self->{'_delimiter'}.$start. $self->{'_delimiter'}.$stop;
      $self->{'_data'}->{$key} = $data;
      if ($self->{'_debug'}) { print "Stop: $key".$self->{'_delimiter'}.$data."\n"; }
   }
}

sub stopall {
   $self = shift;
   my ($nick, $userhost) = @_;
   $nick = $self->{'_tools'}->realnick($nick);
   foreach $channel (split(/;/,$self->{'_config'}->get('IRCCHANNEL'))) {
      $channel = lc($channel);
      my $key = $channel. $self->{'_delimiter'}.$nick;
      if(defined($self->{'_data'}->{$key}) && ($self->{'_data'}->{$key} ne "")) {
         $self->stop($channel, $nick, $userhost);
      }
   }
}

sub stopstartall {
   $self = shift;
   my ($nick, $userhost, $newnick) = @_;
   $nick = $self->{'_tools'}->realnick($nick);
   $newnick = $self->{'_tools'}->realnick($newnick);
   foreach my $channel (split(/;/,$self->{'_config'}->get('IRCCHANNEL'))) {
      $channel = lc($channel);
      my $key = $channel. $self->{'_delimiter'}.$nick;
      if(defined($self->{'_data'}->{$key}) && ($self->{'_data'}->{$key} ne "")) {
         $self->stop($channel, $nick, $userhost);
         $self->start($channel, $newnick, $userhost);
      }
   }
}

sub seen {
   $self = shift;
   my ($reply, $nick, $text) = @_;
   my $cmd = uc($self->{'_config'}->get('COMMANDPREFIX')."SEEN");
   my $search = substr(uc($text),0,length($cmd));
   if ($search eq $cmd) {
      $text = substr($text,length($cmd));
      if ($reply eq "") {
         $self->seen_all($nick, split(/ /,$text));
      } else {
         $self->seen_channel($reply, split(/ /,$text));
      }
   } else {
      $cmd = uc($self->{'_config'}->get('COMMANDPREFIX')."HELP");
      $search = substr(uc($text),0,length($cmd));
      if ($search eq $cmd) {
         $_ = substr($text,length($cmd));
         s/^\s+//;
         s/\s+$//;
         $self->help($nick,$_);
      }
   }
}

sub seen_channel {
   $self = shift;
   my $reply = shift;
   foreach $nick (@_) {
      my $key = $reply. $self->{'_delimiter'}.lc($nick);
      if(defined($self->{'_data'}->{$key}) && ($self->{'_data'}->{$key} ne "")) {
         my ($userhost, $start, $stop) = split( $self->{'_delimiter'},$self->{'_data'}->{$key});
         my $msg = "";
         if ($stop ne "") {
            $msg = "Last time i saw $nick it was '".localtime($start)."'";
            my $online = int(($stop-$start)/60+0.5);
            $msg = $msg." for $online";
            if ($online == 1) {
               $msg = $msg." minute";
            } else {
               $msg = $msg." minutes";
            }
         } else {
            $msg = "$nick is online since '".localtime($start)."'";
         }
         if (defined($self->{'_connection'})) {
            $self->{'_connection'}->privmsg($reply,$msg.".");
         }
      }
   }
}

sub seen_all {
   $self = shift;
   $reply = shift;
   *DATA = $self->{'_data'};
   foreach $nick (@_) {
      foreach my $key (keys %DATA) {
         my ($lchannel, $lnick) = split($self->{'_delimiter'},$key);
         if ($nick eq $lnick) {
            my ($userhost, $start, $stop) = split( $self->{'_delimiter'},$self->{'_data'}->{$key});
            my $msg = "Last time i saw $nick it was '".localtime($start)."'";
            $msg = $msg." on $lchannel";
            if ($stop ne "") {
               my $online = int(($stop-$start)/60+0.5);
               $msg = $msg." for $online";
               if ($online == 1) {
                  $msg = $msg." minute";
               } else {
                  $msg = $msg." minutes";
               }
            }
            if (defined($self->{'_connection'})) {
               $self->{'_connection'}->privmsg($reply,$msg.".");
            }
         }
      }
   }
}

sub help {
   $self = shift;
   my $nick = shift;
   if (scalar(@_) > 0) {
      $cmd = lc(shift);
   }
   if (length($cmd) > 0) {
      if ($cmd eq "seen") {
         $self->{'_connection'}->privmsg($nick,"seen <nick> - Shows the last presence of <nick> on this channel if you ask public or a list of channels where the bot is on.");
      }
   } else {
      # print list
      $self->{'_connection'}->privmsg($nick,"seen            - Last presence of someone.");
   }
}
