##################################################################
# Description:   Storing messages for nicks who aren't currently #
#                online.                                         #
# Author:        Norbert Kuemin <norbert.kuemin@gmx.net>         #
# Copyright:     Copyright 2000 by Norbert Kuemin                #
##################################################################
# Version   Date         Description                             #
#    1.00   26.07.2000   Initial coding                          #
##################################################################

package modules::mail;

use NoK::tools;
use IO::File;
use File::Copy;

$DEBUG = 0;
$VERSION = "1.00";
$NAME = "Mail";
$MODULE = "modules::mail";

sub new {
   my $proto = shift;

   my $conn = undef;
   if (scalar(@_) > 0) {
      $conn = shift;
   }

   my $config = undef;
   if (scalar(@_) > 0) {
      $config = shift;
   }

   my $eventhandler = undef;
   if (scalar(@_) > 0) {
      $eventhandler = shift;
   }

   my $modulehandler = undef;
   if (scalar(@_) > 0) {
      $modulehandler = shift;
   }

   my $self = {
         '_events'        => 'msg userident dcc dcc_close',
         '_config'        => $config,
         '_connection'    => $conn,
         '_eventhandler'  => $eventhandler,
         '_modulehandler' => $modulehandler,
         '_user'          => undef,
         '_filename'      => 'mail.index',
         '_data'          => {},
         '_delimiter'     => "\t",
         '_tools'         => new NoK::tools,
         '_prepared'      => {}
   };
   bless $self, $proto;

   if (defined($config)) {
      if (defined($config->get('MAILFILE'))) {
         $self->{'_filename'} = $config->get('MAILFILE');
      }
   }

   $self->load();

   if ($DEBUG) { print "CREATE $self\n"; }

   return $self;
}

sub DESTROY {
   if ($DEBUG) { print "DESTROY $self\n"; }
}

sub on_msg {
   $self = shift;
   my ($nick, $userhost, $text) = @_;
   $nick = $self->{'_tools'}->realnick($nick);
   # Security check (if possible)
   my @words = split(/ /,$text);
   my $cmd = uc(shift(@words));
   if ($cmd eq uc($self->{'_config'}->get('COMMANDPREFIX')."MAIL")) {
      $cmd = uc(shift(@words));
      if ($cmd eq "MSGSEND") { $self->send_message($nick,@words); }
      if ($cmd eq "MSGLIST") { $self->list_message($nick,@words); }
      if ($cmd eq "MSGREAD") { $self->read_message($nick,@words); }
      if ($cmd eq "MSGDEL") { $self->delete_message($nick,@words); }
      if ($cmd eq "FILESEND") { $self->send_file($nick,@words); }
      if ($cmd eq "FILELIST") { $self->list_file($nick,@words); }
      if ($cmd eq "FILEGET") { $self->get_file($nick,@words); }
      if ($cmd eq "FILEDEL") { $self->delete_file($nick,@words); }
   }
   if ($cmd eq uc($self->{'_config'}->get('COMMANDPREFIX')."HELP")) {
      my $userhandler = $self->{'_modulehandler'}->get("modules::user");
      if (defined($userhandler)) {
         my $user = $userhandler->get_user($nick);
         if (length($user) != 0) {
            $self->help($nick,@words);
         }
      } else {
         $self->help($nick,@words);
      }
   }
}

sub on_userident {
   $self = shift;
   my ($nick,$user) = @_;
   my ($msg,$file) = $self->count($user);
   $self->{'_connection'}->privmsg($nick,"MAIL: You have $msg mails and $file files waiting.");
}

sub on_dcc {
   $self = shift;
   my $nick = shift;
   $nick = $self->{'_tools'}->realnick($nick);
   my $userhost = shift;
   my ($type, $file, $address, $port, $filesize) = split(/ /,shift);
   if ($type ne "SEND") {
      return;
   }
   my $user = lc($nick);
   if (defined($userhandler)) {
      $user = $userhandler->get_user($nick);
      if (length($user) == 0) {
         $self->{'_connection'}->privmsg($nick,"MAIL: Please register / identify first ($nick).");
         return;
      }
   }
   if (!defined($self->{'_prepared'}->{$user})) {
      $self->{'_connection'}->privmsg($nick,"MAIL: Please use first MAIL FILESEND <nick>.");
      return;
   }
   my $key = time().$self->{'_delimiter'}.$self->{'_prepared'}->{$user};
   my $data = "FILE".$self->{'_delimiter'}.$user.$self->{'_delimiter'}.$file;
   my $newfile = $self->{'_config'}->get('UPLOADDIR')."/".$key;
   $self->{'_fh'}->{$user} = IO::File->new(">$newfile");
   if (defined($self->{'_fh'}->{$user})) {
      $self->{'_connection'}->new_get($nick, $key, $address, $port, $filesize, $self->{'_fh'}->{$user});
   }
   delete $self->{'_prepared'}->{$user};
   $self->{'_data'}->{$key} = $data;
   $self->save();
}

sub on_dcc_close {
   $self = shift;
   my $nick = shift;
   $nick = $self->{'_tools'}->realnick($nick);
   my $userhost = shift;
   my $args = shift;
   my $user = lc($nick);
   my $userhandler = $self->{'_modulehandler'}->get("modules::user");
   if (defined($userhandler)) {
      $user = $userhandler->get_user($nick);
      if (length($user) == 0) {
         return;
      }
   }
   if (defined($self->{'_fh'}->{$user})) {
      $self->{'_fh'}->{$user}->close();
      delete $self->{'_fh'}->{$user};
   }
   $self->{'_connection'}->privmsg($nick,"File sent.");
}

sub get_version {
   return $VERSION;
}

sub get_name {
   return $NAME;
}

sub get_module {
   return $MODULE;
}

sub get_events {
   $self = shift;
   return $self->{'_events'};
}

sub load {
   $self = shift;
   # Loading data
   if (-r $self->{'_filename'}) {
      open FD, $self->{'_filename'} or die "Seen: Error reading '".$self->{'_filename'}."'.\n";
      while (<FD>) {
         chomp;
         my ($date,$to,$type,$from,$text) = split($self->{'_delimiter'});
         my $key = $date.$self->{'_delimiter'}.$to;
         my $data = $type.$self->{'_delimiter'}.$from.$self->{'_delimiter'}.$text;
         $self->{'_data'}->{$key} = $data;
      }
      close FD;
   }
}

sub save {
   $self = shift;
   open FD, ">".$self->{'_filename'};
   *DATA = $self->{'_data'};
   foreach my $key (keys %DATA) {
      print FD $key. $self->{'_delimiter'}.$DATA{$key}."\n";
   }
   close FD;
}

sub count {
   $self = shift;
   $user = shift;
   *DATA = $self->{'_data'};
   my $file = 0;
   my $msg = 0;
   foreach my $key (keys %DATA) {
      my ($cdate,$cuser) = split($self->{'_delimiter'},$key);
      my ($ctype,$cfrom,$ctext) = split($self->{'_delimiter'},$DATA{$key});
      if ($cuser eq $user) {
         if ($ctype eq "MSG") {
            $msg++;
         } else {
            $file++;
         }
      }
   }
   return ($msg,$file);
}

sub send_message {
   $self = shift;
   my $nick = shift;
   my $user = lc($nick);
   my $touser = shift;
   my $userhandler = $self->{'_modulehandler'}->get("modules::user");
   if (defined($userhandler)) {
      $user = $userhandler->get_user($nick);
      if (length($user) == 0) {
         $self->{'_connection'}->privmsg($nick,"MAIL: Please register / identify first.");
         return;
      }
      if ($userhandler->exists(lc($touser)) == 0) {
         $self->{'_connection'}->privmsg($nick,"MAIL: User '$user' doesn't exists.");
         return;
      }
      $touser = lc($touser);
   }
   my $key = time().$self->{'_delimiter'}.$touser;
   my $data = "MSG".$self->{'_delimiter'}.$user.$self->{'_delimiter'}.join(" ",@_);
   $self->{'_data'}->{$key} = $data;
   $self->save();
   $self->{'_connection'}->privmsg($nick,"Message sent.");
}

sub list_message {
   $self = shift;
   my $nick = shift;
   *DATA = $self->{'_data'};
   my $msg = 0;
   my $header = 1;
   foreach my $key (sort(keys %DATA)) {
      my ($cdate,$cuser) = split($self->{'_delimiter'},$key);
      my ($ctype,$cfrom,$ctext) = split($self->{'_delimiter'},$DATA{$key});
      if ($cuser eq $user) {
         if ($ctype eq "MSG") {
            $msg++;
            if ($header == 1) {
               $self->{'_connection'}->privmsg($nick,"Nr.   Date (GMT)                 From");
               $header = 0;
            }
            $self->{'_connection'}->privmsg($nick,substr("   ".$msg,-3)."   ".gmtime($cdate)."   ".$cuser);
         }
      }
   }
   if ($header == 1) {
      $self->{'_connection'}->privmsg($nick,"No messages found.");
   }
}

sub read_message {
   $self = shift;
   my $nick = shift;
   my $msgnr = shift;
   *DATA = $self->{'_data'};
   my $msg = 0;
   foreach my $key (sort(keys %DATA)) {
      my ($cdate,$cuser) = split($self->{'_delimiter'},$key);
      my ($ctype,$cfrom,$ctext) = split($self->{'_delimiter'},$DATA{$key});
      if ($cuser eq $user) {
         if ($ctype eq "MSG") {
            $msg++;
            if ($msg == $msgnr) {
               $self->{'_connection'}->privmsg($nick,"From: $cfrom");
               $self->{'_connection'}->privmsg($nick,"To:   $cuser");
               $self->{'_connection'}->privmsg($nick,"Date: ".gmtime($cdate));
               $self->{'_connection'}->privmsg($nick,"Text: $ctext");
               return;
            }
         }
      }
   }
   $self->{'_connection'}->privmsg($nick,"MAIL: Error message #$msgnr not found.");
}

sub delete_message {
   $self = shift;
   my $nick = shift;
   my $msgnr = shift;
   *DATA = $self->{'_data'};
   my $msg = 0;
   my $deleted = 0;
   foreach my $key (sort(keys %DATA)) {
      my ($cdate,$cuser) = split($self->{'_delimiter'},$key);
      my ($ctype,$cfrom,$ctext) = split($self->{'_delimiter'},$DATA{$key});
      if ($cuser eq $user) {
         if ($ctype eq "MSG") {
            $msg++;
            if (($msg == $msgnr) || ($msgnr eq "*")) {
               $deleted = 1;
               my $key = $cdate.$self->{'_delimiter'}.$cuser;
               delete $self->{'_data'}->{$key};
            }
         }
      }
   }
   if ($deleted == 0) {
      $self->{'_connection'}->privmsg($nick,"MAIL: Error message #$msgnr not found.");
   } else {
      $self->save();
   }
}

sub list_file {
   $self = shift;
   my $nick = shift;
   *DATA = $self->{'_data'};
   my $file = 0;
   my $header = 1;
   foreach my $key (sort(keys %DATA)) {
      my ($cdate,$cuser) = split($self->{'_delimiter'},$key);
      my ($ctype,$cfrom,$ctext) = split($self->{'_delimiter'},$DATA{$key});
      if ($cuser eq $user) {
         if ($ctype eq "FILE") {
            $file++;
            if ($header == 1) {
               $self->{'_connection'}->privmsg($nick,"Nr.   Date (GMT)                 From               Size    Filename");
               $header = 0;
            }
	    my $filename = $self->{'_config'}->get('UPLOADDIR')."/".$key;
            if (-r $filename) {
               my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = stat($filename);
               my $ext = 0;
               while ($size > 1024) {
                  $ext++;
                  $size = $size / 1024;
               }
               if ($size >= 100) {
                  $size = " ".int($size);
               } else {
                  $size = substr($size,0,4);
               }
               if ($ext == 0) { $size = $size." "; }
               elsif ($ext == 1) { $size = $size."K"; }
               elsif ($ext == 2) { $size = $size."M"; }
               elsif ($ext == 3) { $size = $size."G"; }
               elsif ($ext == 4) { $size = $size."T"; }
               else { $size = $size."?"; }
               $size = substr("     ".$size,-5);
               $self->{'_connection'}->privmsg($nick,substr("   ".$file,-3)."   ".gmtime($cdate)."   ".substr($cuser."                ",0,16)."   ".$size."   ".$ctext);
            }
         }
      }
   }
   if ($header == 1) {
      $self->{'_connection'}->privmsg($nick,"No files found.");
   }
}

sub send_file {
   $self = shift;
   my $nick = shift;
   my $touser = lc(shift);
   my $user = lc($nick);
   my $userhandler = $self->{'_modulehandler'}->get("modules::user");
   if (defined($userhandler)) {
      $user = $userhandler->get_user($nick);
      if (length($user) == 0) {
         $self->{'_connection'}->privmsg($nick,"MAIL: Please register / identify first.");
         return;
      }
      if ($userhandler->exists(lc($touser)) == 0) {
         $self->{'_connection'}->privmsg($nick,"MAIL: User '$user' doesn't exists.");
         return;
      }
   }
   $self->{'_prepared'}->{$user} = $touser;
   $self->{'_connection'}->privmsg($nick,"Please send now the file with DCC. ($user,$touser)");
}

sub get_file {
   $self = shift;
   my $nick = shift;
   my $filenr = shift;
   *DATA = $self->{'_data'};
   my $file = 0;
   my $olddir = $ENV{PWD};
   chdir $self->{'_config'}->get('UPLOADDIR');
   foreach my $key (sort(keys %DATA)) {
      my ($cdate,$cuser) = split($self->{'_delimiter'},$key);
      my ($ctype,$cfrom,$ctext) = split($self->{'_delimiter'},$DATA{$key});
      if ($cuser eq $user) {
         if ($ctype eq "FILE") {
            $file++;
            if (($file == $filenr) || ($filenr eq "*")) {
               my $localfile = $key;
               my $remotefile = $ctext;
               copy($localfile,$remotefile);
               my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = stat($remotefile);
               $self->{'_connection'}->new_send($nick,$remotefile,$blocks);
               unlink($remotefile);
            }
         }
      }
   }
}

sub delete_file {
   $self = shift;
   my $nick = shift;
   my $filenr = shift;
   *DATA = $self->{'_data'};
   my $file = 0;
   my $deleted = 0;
   foreach my $key (sort(keys %DATA)) {
      my ($cdate,$cuser) = split($self->{'_delimiter'},$key);
      my ($ctype,$cfrom,$ctext) = split($self->{'_delimiter'},$DATA{$key});
      if ($cuser eq $user) {
         if ($ctype eq "FILE") {
            $file++;
            if (($file == $filenr) || ($filenr eq "*")) {
               $deleted = 1;
               my $key = $cdate.$self->{'_delimiter'}.$cuser;
               delete $self->{'_data'}->{$key};
               unlink($self->{'_config'}->get('UPLOADDIR')."/".$key);
            }
         }
      }
   }
   if ($deleted == 0) {
      $self->{'_connection'}->privmsg($nick,"MAIL: Error file #$filenr not found.");
   } else {
      $self->save();
   }
}

sub help {
   $self = shift;
   my $nick = shift;
   my $cmd = uc(shift);
   if (length($cmd) == 0) {
      # print list
      $self->{'_connection'}->privmsg($nick,"filedel         - Delete a file.");
      $self->{'_connection'}->privmsg($nick,"fileget         - Download a file.");
      $self->{'_connection'}->privmsg($nick,"filelist        - List of uploaded files.");
      $self->{'_connection'}->privmsg($nick,"filesend        - Upload a file for a the specified user.");
      $self->{'_connection'}->privmsg($nick,"msgdel          - Delete a message.");
      $self->{'_connection'}->privmsg($nick,"msgget          - Read a message.");
      $self->{'_connection'}->privmsg($nick,"msglist         - List of messages.");
      $self->{'_connection'}->privmsg($nick,"msgsend         - Send a <text> to <nick>.");
   }
   if ($cmd eq "MSGSEND") {
      $self->{'_connection'}->privmsg($nick,"msgsend <nick> <text> - Send a <text> to <nick>.");
   }
   if ($cmd eq "MSGLIST") {
      $self->{'_connection'}->privmsg($nick,"msglist - Lists all messages for your nick.");
   }
   if ($cmd eq "MSGREAD") {
      $self->{'_connection'}->privmsg($nick,"msgget - Read message #.");
   }
   if ($cmd eq "MSGDEL") {
      $self->{'_connection'}->privmsg($nick,"msgdel - Delete message #.");
   }
   if ($cmd eq "FILESEND") {
      $self->{'_connection'}->privmsg($nick,"filesend <nick> - Send a file to <nick>.");
   }
   if ($cmd eq "FILELIST") {
      $self->{'_connection'}->privmsg($nick,"filelist - Lists all files for your nick.");
   }
   if ($cmd eq "FILEREAD") {
      $self->{'_connection'}->privmsg($nick,"fileget - Download file #.");
   }
   if ($cmd eq "FILEDEL") {
      $self->{'_connection'}->privmsg($nick,"filedel - Delete file #.");
   }
}
