##################################################################
# Description:   Logging activities on IRC                       #
# Author:        Norbert Kuemin <norbert.kuemin@gmx.net>         #
# Copyright:     Copyright 2000 by Norbert Kuemin                #
##################################################################
# Version   Date         Description                             #
#     1.0   23.07.2000   Initial coding                          #
##################################################################

package modules::logtext;

$VERSION = "1.0";
$NAME = "LogText";
$MODULE = "modules::logtext";

sub new {
   my $proto = shift;

   my $conn = undef;
   if (scalar(@_) > 0) {
      $conn = shift;
   }

   my $config = undef;
   if (scalar(@_) > 0) {
      $config = shift;
   }

   my $eventhandler = undef;
   if (scalar(@_) > 0) {
      $eventhandler = shift;
   }

   my $modulehandler = undef;
   if (scalar(@_) > 0) {
      $modulehandler = shift;
   }

   my $self = {
         '_msgprefix'     => '***',
         '_privatfile'    => 'privat',
         '_config'        => $config,
         '_connection'    => $conn,
         '_eventhandler'  => $eventhandler,
         '_modulehandler' => $modulehandler,
         '_events'        => 'init join part quit nick msg public names ping action disconnect topic mode kick'
   };

   bless $self, $proto;

   if (defined($config)) {
      if (defined($config->get('LOGMSGPREFIX'))) {
         $self->{'_msgprefix'} = $config->get('LOGMSGPREFIX');
      }
      if (defined($config->get('LOGPRIVATFILE'))) {
         $self->{'_privatfile'} = $config->get('LOGPRIVATFILE');
      }
   }

   return $self;
}

sub write_file {
   $self = shift;
   my ($logdir,$channel,$text) = @_;
   my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
   $year += 1900;
   $mon++;
   my $file = $logdir . $year;
   if ($mon > 9) {
      $file = $file . $mon;
   } else {
      $file = $file . "0" . $mon;
   }
   if ($mday > 9) {
      $file = $file . $mday;
   } else {
      $file = $file . "0" . $mday;
   }
   $file = $file . "." . uc($channel) . ".txt";
   my $time = "";
   if ($hour > 9) {
      $time = $hour;
   } else {
      $time = "0" . $hour;
   }
   $time =  $time . ":";
   if ($min > 9) {
      $time = $time . $min;
   } else {
      $time = $time . "0" . $min;
   }
   $time =  $time . ":";
   if ($sec > 9) {
      $time = $time . $sec;
   } else {
      $time = $time . "0" . $sec;
   }
   open FD, ">>".$file;
   print FD $time." ".$text."\n";
   close FD;
}

sub write_log {
   my $self = shift;
   my ($channel, $msg) = @_;
   my $logdir = "";
   if (defined($self->{'_config'})) {
      $logdir = $self->{'_config'}->get('LOGDIR');
   } else {
      $logdir = "";
   }
   if (length($logdir) > 0) {
      $self->write_file($logdir,$channel,$msg);
   } else {
      if ($channel ne $self->{'_privatfile'}) {
         if (index($msg,$channel) == -1) { print "$channel "; }
      }
      print "$msg\n";
   }
}

sub on_init {
   $self = shift;
   $text = shift;
   my ($msg) = $self->{'_msgprefix'}." ".$text;
   $self->write_log($self->{'_privatfile'}, $msg);
}

sub on_join {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   my ($msg) = $self->{'_msgprefix'}." ".$nick." (".$userhost.") has joined channel ".$channel;
   $self->write_log($channel, $msg);
}

sub on_part {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   my ($msg) = $self->{'_msgprefix'}." ".$nick." (".$userhost.") has left channel ".$channel;
   $self->write_log($channel, $msg);
}

sub on_quit {
   $self = shift;
   my ($nick, $userhost, $reason) = @_;
   my ($msg) = self->{'_msgprefix'}." ".$nick." (".$userhost.") has quit (".$reason.")";
   $self->write_log($self->{'_privatfile'}, $msg);
}

sub on_nick {
   $self = shift;
   my ($nick, $userhost, $newnick) = @_;
   my $msg = $self->{'_msgprefix'}." $nick is now known as $newnick";
   $self->write_log($self->{'_privatfile'}, $msg);
}

sub on_msg {
   $self = shift;
   my ($nick, $userhost, $text) = @_;
   my ($msg) = "<$nick> $text";
   $self->write_log($self->{'_privatfile'}, $msg);
}

sub on_public {
   $self = shift;
   my ($channel, $nick, $userhost, $text) = @_;
   my $msg = "<$nick> $text";
   $self->write_log($channel, $msg);
}

sub on_names {
   $self = shift;
   ($channel, @list) = @_;
   my $msg = $self->{'_msgprefix'}." Users on $channel: @list";
   $self->write_log($channel, $msg);
}

sub on_ping {
   $self = shift;
   my ($nick, $userhost, $args) = @_;
   my $msg = $self->{'_msgprefix'}." CTCP PING request from $nick received (" .
      join (' ', ($args)) . ").";
   $self->write_log($self->{'_privatfile'}, $msg);
}

sub on_action {
   $self = shift;
   ($channel, $nick, $userhost, $action) = @_;
   my $msg = "* $nick $action";
   $self->write_log($channel, $msg);
}

sub on_disconnect {
   my $self = shift;
   my ($server, $reason) = @_;
   my $msg = $self->{'_msgprefix'}." Disconnected from $server ($reason)";
   $self->write_log($self->{'_privatfile'}, $msg);
}

sub on_topic {
   $self = shift;
   my ($channel, $topic) = @_;
   $self->write_log($channel, $self->{'_msgprefix'}." Change the topic for $channel to \"$topic\".");
}

sub on_mode {
   $self = shift;
   my ($channel, $nick, $userhost, $mode, $arg) = @_;
   $self->write_log($channel, $self->{'_msgprefix'}." $channel mode [$mode $arg] set by $nick");
}

sub on_kick {
   $self = shift;
   my ($channel, $nick, $userhost, $user, $reason) = @_;
   $self->write_log($channel, $self->{'_msgprefix'}." $user was kicked from $channel by $nick [$reason]");
}

sub get_version {
   return $VERSION;
}

sub get_name {
   return $NAME;
}

sub get_module {
   return $MODULE;
}

sub get_events {
   $self = shift;
   return $self->{'_events'};
}
