##################################################################
# Description:   Logging activities on IRC                       #
# Author:        Norbert Kuemin <norbert.kuemin@gmx.net>         #
# Copyright:     Copyright 2000 by Norbert Kuemin                #
##################################################################
# Version   Date         Description                             #
#    1.00   23.07.2000   Initial coding                          #
##################################################################

package modules::announce;

use POSIX qw(mktime strftime tzset);

$VERSION = "1.00";
$NAME = "Announce";
$MODULE = "modules::announce";

sub new {
   my $proto = shift;

   my $conn = undef;
   if (scalar(@_) > 0) {
      $conn = shift;
   }

   my $config = undef;
   if (scalar(@_) > 0) {
      $config = shift;
   }

   my $eventhandler = undef;
   if (scalar(@_) > 0) {
      $eventhandler = shift;
   }

   my $modulehandler = undef;
   if (scalar(@_) > 0) {
      $modulehandler = shift;
   }

   my $self = {
         '_config'        => $config,
         '_connection'    => $conn,
         '_eventhandler'  => $eventhandler,
         '_modulehandler' => $modulehandler,
         '_events'        => 'join msg',
         '_filename'      => 'announce.list',
         '_delimiter'     => "\t",
         '_data'          => (),
         '_tools'         => undef
   };
   bless $self, $proto;

   if (defined($config)) {
      if (defined($config->get('ANNOUNCEFILE'))) {
         $self->{'_filename'} = $config->get('ANNOUNCEFILE');
      }
   }
   $self->{'_tools'} = new NoK::tools;
   $self->load();
   $ENV{'TZ'}="MET";
   tzset();

   return $self;
}

sub on_join {
   $self = shift;
   my ($channel, $nick, $userhost) = @_;
   *DATA = $self->{'_data'};
   my $count = 0;
   foreach my $line (@DATA) {
      $count++;
      my ($an_channel,$an_until,$an_text) = split($self->{'_delimiter'},$line,3);
      if ((lc($channel) eq lc($an_channel)) && (($an_until >= time()) || ($an_until == 0))) {
         if (defined($self->{'_connection'})) {
            $self->{'_connection'}->notice($nick,$an_text);
         }
      }
   }
}

sub on_msg {
   $self = shift;
   my ($nick, $userhost, $text) = @_;
   $nick = $self->{'_tools'}->realnick($nick);
   # Security check (if possible)
   my @words = split(/ /,$text);
   my $cmd = uc(shift(@words));
   if ($cmd eq uc($self->{'_config'}->get('COMMANDPREFIX')."ANNOUNCE")) {
      my $userhandler = $self->{'_modulehandler'}->get("modules::user");
      if (defined($userhandler)) {
         if ($userhandler->checkmode($nick,"m") == 0) {
            $userhandler->error($nick,"ANNOUNCE","You cannot control the announcements oft this bot.");
            return;
         }
      }
      $cmd = uc(shift(@words));
      if ($cmd eq "LIST") {
         $self->list($nick);
      }
      if ($cmd eq "ADD") {
         my $date = shift(@words);
         my $utime = 0;
         if (uc($date) eq "FOREVER") {
            $utime = 0;
         } else {
            my $time = shift(@words);
            my ($year,$month,$day) = split("-",$date,3);
            my ($hour,$minute,$second) = split(":",$time,3);
            $year = $year - 1900;
            $month--;
            $utime = mktime($second,,$minute,$hour,$day,$month,$year);
         }
         my $chan = lc(shift(@words));
         $self->add($nick, $chan, $utime, join(" ",@words));
      }
      if ($cmd eq "DEL") {
         my $nr = shift(@words);
         $self->del($nick, $nr);
      }
      if ($cmd eq "RELOAD") {
         $self->reload($nick);
      }
   }
   if ($cmd eq uc($self->{'_config'}->get('COMMANDPREFIX')."HELP")) {
      my $userhandler = $self->{'_modulehandler'}->get("modules::user");
      if (defined($userhandler)) {
         if ($userhandler->checkmode($nick,"m") != 0) {
            $self->help($nick,@words);
         }
      } else {
         $self->help($nick,@words);
      }
   }
}


sub get_version {
   return $VERSION;
}

sub get_name {
   return $NAME;
}

sub get_module {
   return $MODULE;
}

sub get_events {
   $self = shift;
   return $self->{'_events'};
}

sub load {
   $self = shift;
   if (-r $self->{'_filename'}) {
      open FD, $self->{'_filename'} or die "Seen: Error reading '".$self->{'_filename'}."'.\n";
      my $count = -1;
      while (<FD>) {
         chomp();
         $count++;
         $self->{'_data'}->[$count] = $_;
      }
      close FD;
   }
}

sub save {
   $self = shift;
   open FD, ">".$self->{'_filename'};
   *DATA = $self->{'_data'};
   foreach my $line (@DATA) {
      print FD $line."\n";
   }
   close FD;
}

sub add {
   $self = shift;
   my ($nick, $chan, $utime, $text) = @_;
   *DATA = $self->{'_data'};
   push @DATA, $chan.$self->{'_delimiter'}.$utime.$self->{'_delimiter'}.$text;
   $self->save();
   $self->{'_connection'}->notice($nick,"Announcement added.");
}

sub del {
   $self = shift;
   my ($nick, $nr) = @_;
   if ($nr > 0) {
      $nr--;
      *DATA = $self->{'_data'};
      splice(@DATA,$nr,1);
      $self->save();
      $self->{'_connection'}->notice($nick,"Announcement deleted.");
   }
}

sub list {
   $self = shift;
   my ($nick) = @_;
   *DATA = $self->{'_data'};
   my $count = 0;
   if (scalar(@DATA) > 0) {
      for ($n=0;$n<scalar(@DATA);$n++) {
         my ($an_channel,$an_until,$an_text) = split($self->{'_delimiter'},$DATA[$n],3);
         my $time = "forever";
         if ($an_until > 0) { 
            $time = strftime("%Y-%m-%d %H:%M:%S",localtime($an_until));
         }
         my $nr = $n+1;
         $self->{'_connection'}->notice($nick,$nr.". ".$an_channel." ".$an_text." (".$time.")");
      }
   } else {
      $self->{'_connection'}->notice($nick,"No announcements.");
   }
}

sub help {
   $self = shift;
}

sub reload {
   $self = shift;
   my ($nick) = @_;
   *DATA = $self->{'_data'};
   @DATA = ();
   $self->load();
   $self->{'_connection'}->notice($nick,"Announcement reloaded.");
}